const modeSelect = document.getElementById("mode");
const onceSettings = document.getElementById("once-settings");
const repeatSettings = document.getElementById("repeat-settings");
const calendarSettings = document.getElementById("calendar-settings");
const profileInput = document.getElementById("profileName");
const status = document.getElementById("status");
const tasksList = document.getElementById("tasksList");
const tasksCount = document.getElementById("tasksCount");
const totalTasks = document.getElementById("totalTasks");
const runsCount = document.getElementById("runsCount");
const runsPerTrigger = document.getElementById("runsPerTrigger");

modeSelect.addEventListener("change", () => {
  onceSettings.style.display = modeSelect.value === "once" ? "block" : "none";
  repeatSettings.style.display = modeSelect.value === "repeat" ? "block" : "none";
  calendarSettings.style.display = modeSelect.value === "calendar" ? "block" : "none";
});

window.addEventListener("DOMContentLoaded", () => {
  onceSettings.style.display = "block";
});

runsPerTrigger.addEventListener("change", () => {
  runsCount.textContent = runsPerTrigger.value;
});

function convertTriggerToMinutes(days, hours, minutes, seconds) {
  return (days * 24 * 60) + (hours * 60) + parseInt(minutes) + (parseInt(seconds) / 60);
}

function convertDelayToMs(seconds, milliseconds) {
  return (parseInt(seconds) * 1000) + parseInt(milliseconds);
}

function updateTasksDisplay() {
  chrome.storage.local.get(["activeTasks"], (data) => {
    const tasks = data.activeTasks || [];
    tasksCount.textContent = tasks.length;
    totalTasks.textContent = tasks.length;

    if (tasks.length === 0) {
      tasksList.innerHTML = '<div class="no-tasks">אין משימות פעילות</div>';
      return;
    }

    tasksList.innerHTML = tasks.map((task, index) => {
      const nextDate = new Date(task.nextRun);
      const now = new Date();
      const timeLeft = Math.max(0, (nextDate - now) / 1000);
      
      const days = Math.floor(timeLeft / 86400);
      const hours = Math.floor((timeLeft % 86400) / 3600);
      const mins = Math.floor((timeLeft % 3600) / 60);
      const secs = Math.floor(timeLeft % 60);

      const timeStr = days > 0 
        ? `${days}d ${hours}h ${mins}m` 
        : `${hours}h ${mins}m ${secs}s`;

      return `
        <div class="task-item">
          <div class="task-info">
            <div class="task-name">📍 ${task.profileName}</div>
            <div class="task-time">
              🕐 ${task.mode === 'repeat' ? '🔄 ' : ''}הבאה בעוד: ${timeStr}
              ${task.runs > 1 ? ` | ${task.runs}x` : ''}
            </div>
          </div>
          <button class="task-remove" data-index="${index}">✕</button>
        </div>
      `;
    }).join('');

    setTimeout(updateTasksDisplay, 1000);
  });
}

function removeTask(index) {
  chrome.storage.local.get(["activeTasks"], (data) => {
    const tasks = data.activeTasks || [];
    const taskToRemove = tasks[index];
    
    if (taskToRemove.alarmId) {
      chrome.alarms.clear(taskToRemove.alarmId);
    }
    
    tasks.splice(index, 1);
    chrome.storage.local.set({ activeTasks: tasks });
    updateTasksDisplay();
  });
}

window.removeTask = removeTask;

updateTasksDisplay();

// Event delegation לכפתורי ההסרה
document.addEventListener("click", (e) => {
  if (e.target.classList.contains("task-remove")) {
    const index = parseInt(e.target.dataset.index);
    removeTask(index);
  }
});

document.getElementById("startBtn").addEventListener("click", () => {
  const profileName = profileInput.value.trim() || "ללא שם";
  const mode = modeSelect.value;
  const runs = parseInt(runsPerTrigger.value) || 1;

  let minutesDelay = 0;
  let periodMinutes = 0;
  let alarmId = `autoRun_${Date.now()}`;

  if (mode === "once") {
    const days = parseInt(document.getElementById("onceDays").value) || 0;
    const hours = parseInt(document.getElementById("onceHours").value) || 0;
    const minutes = parseInt(document.getElementById("onceMinutes").value) || 1;
    const seconds = parseInt(document.getElementById("onceSeconds").value) || 0;
    const delaySeconds = parseInt(document.getElementById("onceDelaySeconds").value) || 0;
    const delayMilliseconds = parseInt(document.getElementById("onceDelayMilliseconds").value) || 500;
    const delayBetweenRuns = convertDelayToMs(delaySeconds, delayMilliseconds);
    minutesDelay = convertTriggerToMinutes(days, hours, minutes, seconds);
    chrome.alarms.create(alarmId, { delayInMinutes: minutesDelay });
    status.textContent = `✅ משימה נוספה: ${profileName} (${runs}x) בעוד ${days}d ${hours}h ${minutes}m ${seconds}s | עיכוב: ${delaySeconds}s ${delayMilliseconds}ms`;
    status.className = "success";
  } 
  else if (mode === "repeat") {
    const days = parseInt(document.getElementById("repeatDays").value) || 0;
    const hours = parseInt(document.getElementById("repeatHours").value) || 0;
    const minutes = parseInt(document.getElementById("repeatMinutes").value) || 1;
    const seconds = parseInt(document.getElementById("repeatSeconds").value) || 0;
    const delaySeconds = parseInt(document.getElementById("repeatDelaySeconds").value) || 0;
    const delayMilliseconds = parseInt(document.getElementById("repeatDelayMilliseconds").value) || 500;
    const delayBetweenRuns = convertDelayToMs(delaySeconds, delayMilliseconds);
    periodMinutes = convertTriggerToMinutes(days, hours, minutes, seconds);
    chrome.alarms.create(alarmId, { periodInMinutes: periodMinutes });
    status.textContent = `🔄 משימה חוזרת נוספה: ${profileName} (${runs}x) כל ${days}d ${hours}h ${minutes}m ${seconds}s | עיכוב: ${delaySeconds}s ${delayMilliseconds}ms`;
    status.className = "success";
  } 
  else if (mode === "calendar") {
    const timeValue = document.getElementById("calendarTime").value;
    if (!timeValue) {
      status.textContent = "❌ בחר תאריך ושעה";
      status.className = "error";
      return;
    }
    const targetTime = new Date(timeValue).getTime();
    const now = Date.now();
    const delayMinutes = (targetTime - now) / 60000;
    if (delayMinutes > 0) {
      const delaySeconds = parseInt(document.getElementById("calendarDelaySeconds").value) || 0;
      const delayMilliseconds = parseInt(document.getElementById("calendarDelayMilliseconds").value) || 500;
      chrome.alarms.create(alarmId, { delayInMinutes: delayMinutes });
      status.textContent = `📅 משימה בקביעה נוספה: ${profileName} (${runs}x) ב-${timeValue}`;
      status.className = "success";
    } else {
      status.textContent = "❌ בחר זמן בעתיד";
      status.className = "error";
      return;
    }
  }

  const newTask = {
    id: alarmId,
    mode,
    profileName,
    runs,
    nextRun: new Date(Date.now() + (minutesDelay || periodMinutes) * 60000).toISOString(),
    alarmId,
    delayBetweenRuns: mode === "once" 
      ? convertDelayToMs(
          parseInt(document.getElementById("onceDelaySeconds").value) || 0,
          parseInt(document.getElementById("onceDelayMilliseconds").value) || 500
        )
      : mode === "repeat"
      ? convertDelayToMs(
          parseInt(document.getElementById("repeatDelaySeconds").value) || 0,
          parseInt(document.getElementById("repeatDelayMilliseconds").value) || 500
        )
      : 500
  };

  chrome.storage.local.get(["activeTasks"], (data) => {
    const tasks = data.activeTasks || [];
    tasks.push(newTask);
    chrome.storage.local.set({ activeTasks: tasks });
    updateTasksDisplay();
  });
});

document.getElementById("stopBtn").addEventListener("click", () => {
  chrome.storage.local.get(["activeTasks"], (data) => {
    const tasks = data.activeTasks || [];
    tasks.forEach(task => {
      if (task.alarmId) {
        chrome.alarms.clear(task.alarmId);
      }
    });
    chrome.storage.local.set({ activeTasks: [] });
    status.textContent = "🛑 כל ההפעלות נעצרו";
    status.className = "error";
    updateTasksDisplay();
  });
});